/*******************************************************************************************************
 *                                                                                                     *
 *        **********                                                                                   *
 *       ************                                                                                  *
 *      ***        ***                                                                                 *
 *      ***   +++   ***                                                                                *
 *      ***   + +   ***                                                                                *
 *      ***   +                               CHIPCON CC2420 BASIC RF LIBRARY                          *
 *      ***   + +   ***                               Packet reception                                 *
 *      ***   +++   ***                                                                                *
 *      ***        ***                                                                                 *
 *       ************                                                                                  *
 *        **********                                                                                   *
 *                                                                                                     *
 *******************************************************************************************************
 * This file contains functions used to enable and disable for packet reception, and the FIFOP         *
 * interrupt which handles the packet reception. The application must define the                       *
 * basicRfReceivePacket() function, which is called when a packet has been successfully received.      *
 *                                                                                                     *
 * More information can be found in basic_rf.h                                                         *
 *******************************************************************************************************
 * Compiler: AVR-GCC, IAR 3.2, MSP430-GCC                                                                             *
 * Target platform: CC2420DB, CC2420 + any MCU with very few modifications required                    *
 *******************************************************************************************************
 * Revision history:                                                                                   *
 *  $Log: basic_rf_receive.c,v $
 *  Revision 1.4  2004/07/26 11:28:10  mbr
 *  Modified RXFIFO flushing by strobing CC2420_SFLUSHRX
 *
 *  Revision 1.3  2004/03/30 14:59:22  mbr
 *  Release for web
 * 
 *
 *
 *******************************************************************************************************/
#include "include/include.h"
UINT8 spiCnt;
UINT8 i;
//-------------------------------------------------------------------------------------------------------
//  void halRfReceiveOn(void)
//
//  DESCRIPTION:
//      Enables the CC2420 receiver and the FIFOP interrupt. When a packet is received through this
//      interrupt, it will call halRfReceivePacket(...), which must be defined by the application
//-------------------------------------------------------------------------------------------------------
void basicRfReceiveOn(void) {
//	printf("Turning on the receiver...\r\n");
	rfSettings.receiveOn = TRUE;
	FASTSPI_STROBE(CC2420_SRXON);
	FASTSPI_STROBE(CC2420_SFLUSHRX);
	FIFOP_INT_INIT();
    ENABLE_FIFOP_INT();
} // basicRfReceiveOn




//-------------------------------------------------------------------------------------------------------
//  void halRfReceiveOff(void)
//
//  DESCRIPTION:
//      Disables the CC2420 receiver and the FIFOP interrupt.
//-------------------------------------------------------------------------------------------------------
void basicRfReceiveOff(void) {
    rfSettings.receiveOn = FALSE;
	FASTSPI_STROBE(CC2420_SRFOFF);
    DISABLE_FIFOP_INT();
} // basicRfReceiveOff


void basicRfReceivePacket(void){
		UINT8 spiCnt;
//	UINT8 i;
	WORD frameControlField;
	INT8 length;
	BYTE pFooter[2];

    
    // Clean up and exit in case of FIFO overflow, which is indicated by FIFOP = 1 and FIFO = 0
	if((FIFOP_IS_1) && (!(FIFO_IS_1))) {	   
//		printf("Detected FIFO Overflow....\r\n");
	    FASTSPI_STROBE(CC2420_SFLUSHRX);
	    FASTSPI_STROBE(CC2420_SFLUSHRX);
	
	    return;
	}

	// Payload length
	FASTSPI_READ_FIFO_BYTE(length);
// 100	
	length &= BASIC_RF_LENGTH_MASK; // Ignore MSB
//	printf("Length: %x\r\n", length);
    // Ignore the packet if the length is too short
    if (length < BASIC_RF_ACK_PACKET_SIZE) {
//	printf("Detected length < BASIC_RF_ACK_PACKET_SIZE\r\n");
    	FASTSPI_READ_FIFO_GARBAGE(length);

    // Otherwise, if the length is valid, then proceed with the rest of the packet
    } else {
        // Register the payload length
        rfSettings.pRxInfo->length = length - BASIC_RF_PACKET_OVERHEAD_SIZE;
//	printf("Length(payload): %x\r\n",rfSettings.pRxInfo->length);
        // Read the frame control field and the data sequence number
        FASTSPI_READ_FIFO_NO_WAIT((BYTE*) &frameControlField, 2);
	rfSettings.pRxInfo->ackRequest = !!(frameControlField & BASIC_RF_FCF_ACK_BM);
//	printf("AckRequest: %x\r\n", rfSettings.pRxInfo->ackRequest);
    	FASTSPI_READ_FIFO_BYTE(rfSettings.pRxInfo->seqNumber);

		// Is this an acknowledgment packet?
	if ((length == BASIC_RF_ACK_PACKET_SIZE) && (frameControlField == BASIC_RF_ACK_FCF) && (rfSettings.pRxInfo->seqNumber == rfSettings.txSeqNumber)) {
			
 	       	// Read the footer and check for CRC OK
			FASTSPI_READ_FIFO_NO_WAIT((BYTE*) pFooter, 2);

			// Indicate the successful ack reception (this flag is polled by the transmission routine)
			if (pFooter[1] & BASIC_RF_CRC_OK_BM) {
//				printf("ACK received\r\n");
				rfSettings.ackReceived = TRUE;
			}
 
		// Too small to be a valid packet?
		} else if (length < BASIC_RF_PACKET_OVERHEAD_SIZE) {
			FASTSPI_READ_FIFO_GARBAGE(length - 3);
			return;

		// Receive the rest of the packet
		} else {
	
			// Skip the destination PAN and address (that's taken care of by harware address recognition!)
			FASTSPI_READ_FIFO_GARBAGE(4);

			// Read the source address
			FASTSPI_READ_FIFO_NO_WAIT((BYTE*) &rfSettings.pRxInfo->srcAddr, 2);
//			printf("SrcAdd: %x\r\n", rfSettings.pRxInfo->srcAddr);

			// Read the packet payload
			FASTSPI_READ_FIFO_NO_WAIT(rfSettings.pRxInfo->pPayload, rfSettings.pRxInfo->length);
//			printf("Payload[0]: %x\r\n", rfSettings.pRxInfo->pPayload[0]);
//			printf("Payload[1]: %x\r\n", rfSettings.pRxInfo->pPayload[1]);
			// Read the footer to get the RSSI value
			FASTSPI_READ_FIFO_NO_WAIT((BYTE*) pFooter, 2);
//			printf("pFooter:%x\r\n", pFooter);
			rfSettings.pRxInfo->rssi = pFooter[0];

			// Notify the application about the received _data_ packet if the CRC is OK
			if (((frameControlField & (BASIC_RF_FCF_BM)) == BASIC_RF_FCF_NOACK) && (pFooter[1] & BASIC_RF_CRC_OK_BM)) {
//				printf("Notifying the application\r\n");
				rfSettings.pRxInfo = basicRfProcessPacket(rfSettings.pRxInfo);
			}
		}
    }
}
